//GPT-4o-mini DATASET v1.0 Category: Network Packet Monitoring ; Style: Donald Knuth
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <arpa/inet.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#define MAX_PACKETS 100

void packet_handler(u_char *args, const struct pcap_pkthdr *header, const u_char *packet) {
    struct ip *ip_header = (struct ip *)(packet + 14); // Skip Ethernet header
    int ip_header_length = ip_header->ip_hl * 4;
    u_int16_t total_length = ntohs(ip_header->ip_len);
    
    printf("Packet captured: Length: %d bytes\n", total_length);
    printf("Source IP: %s\n", inet_ntoa(ip_header->ip_src));
    printf("Destination IP: %s\n", inet_ntoa(ip_header->ip_dst));
    
    if (ip_header->ip_p == IPPROTO_TCP) {
        struct tcphdr *tcp_header = (struct tcphdr *)(packet + 14 + ip_header_length);
        printf("This packet contains a TCP segment:\n");
        printf("Source Port: %u\n", ntohs(tcp_header->source));
        printf("Destination Port: %u\n", ntohs(tcp_header->dest));
    } else if (ip_header->ip_p == IPPROTO_UDP) {
        struct udphdr *udp_header = (struct udphdr *)(packet + 14 + ip_header_length);
        printf("This packet contains a UDP datagram:\n");
        printf("Source Port: %u\n", ntohs(udp_header->source));
        printf("Destination Port: %u\n", ntohs(udp_header->dest));
    }
    
    printf("----------------------\n");
}

void print_usage() {
    printf("Usage: ./packet_monitor <device>\n");
    printf("Example: ./packet_monitor eth0\n");
}

int main(int argc, char *argv[]) {
    if (argc != 2) {
        print_usage();
        return EXIT_FAILURE;
    }

    char *device = argv[1];
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *handle;

    // Open the capture device
    handle = pcap_open_live(device, BUFSIZ, 1, 1000, errbuf);
    if (handle == NULL) {
        fprintf(stderr, "Could not open device %s: %s\n", device, errbuf);
        return EXIT_FAILURE;
    }

    // Compile the filter expression
    struct bpf_program filter;
    char filter_exp[] = "ip";
    if (pcap_compile(handle, &filter, filter_exp, 0, PCAP_NETMASK_UNKNOWN) == -1) {
        fprintf(stderr, "Could not parse filter %s: %s\n", filter_exp, pcap_geterr(handle));
        return EXIT_FAILURE;
    }

    // Set the filter
    if (pcap_setfilter(handle, &filter) == -1) {
        fprintf(stderr, "Error setting filter %s: %s\n", filter_exp, pcap_geterr(handle));
        return EXIT_FAILURE;
    }

    printf("Listening on device %s...\n", device);

    // Capture packets
    pcap_loop(handle, MAX_PACKETS, packet_handler, NULL);

    // Clean up
    pcap_freecode(&filter);
    pcap_close(handle);
    printf("Packet monitoring terminated.\n");
    
    return EXIT_SUCCESS;
}