//GPT-4o-mini DATASET v1.0 Category: Wireless Network Scanner ; Style: ephemeral
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <arpa/inet.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/ether.h>

#define SNAP_LEN 65535
#define SIZE_ETHERNET 14
#define EAPOL 0x888E
#define WLAN_HDR_LEN 24
#define DOT11_FCS_LEN 4
#define MAX_SSID_LEN 32

typedef struct {
    char ssid[MAX_SSID_LEN + 1];
    uint8_t bssid[6];
    int signal_strength;
} WifiNetwork;

void handle_packet(u_char *args, const struct pcap_pkthdr *header, const u_char *packet);
void print_wifi_networks(WifiNetwork networks[], int count);

int main(int argc, char **argv) {
    char *dev, errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *handle;

    // Check for device argument or use the default interface
    if (argc != 2) {
        dev = pcap_lookupdev(errbuf);
        if (dev == NULL) {
            fprintf(stderr, "Could not determine device: %s\n", errbuf);
            return EXIT_FAILURE;
        }
    } else {
        dev = argv[1];
    }

    // Open the capture device
    handle = pcap_open_live(dev, SNAP_LEN, 1, 1000, errbuf);
    if (handle == NULL) {
        fprintf(stderr, "Could not open device %s: %s\n", dev, errbuf);
        return EXIT_FAILURE;
    }

    // Start capturing packets
    pcap_loop(handle, 0, handle_packet, NULL);

    // Clean up
    pcap_close(handle);
    return EXIT_SUCCESS;
}

void handle_packet(u_char *args, const struct pcap_pkthdr *header, const u_char *packet) {
    static WifiNetwork networks[100];
    static int network_count = 0;

    // Parse the packet
    struct ether_header *eth = (struct ether_header *) packet;
    if (ntohs(eth->ether_type) != ETHERTYPE_PUP && ntohs(eth->ether_type) != ETHERTYPE_IP) {
        return;  // Ignore non-IP packets
    }

    const u_char *wlan_header = packet + SIZE_ETHERNET;
    int wlan_type = wlan_header[0] & 0x0C; // Frame control

    if (wlan_type != 0x00) {
        return;  // Ignore non-management frames
    }

    int ssid_len = wlan_header[37]; // Length of SSID
    if (ssid_len > MAX_SSID_LEN) {
        return;  // Skip if SSID is too long
    }

    WifiNetwork new_network;
    memcpy(new_network.bssid, eth->ether_shost, 6);
    new_network.signal_strength = -100; // Example value; in real code this would be parsed

    snprintf(new_network.ssid, ssid_len + 1, "%s", (char *)(wlan_header + 38));

    // Check for duplicates
    for (int i = 0; i < network_count; i++) {
        if (memcmp(networks[i].bssid, new_network.bssid, 6) == 0) {
            return;  // Network already added
        }
    }

    // Add new network to the list
    networks[network_count++] = new_network;

    // Print found networks every 10 packets
    if (network_count % 10 == 0) {
        print_wifi_networks(networks, network_count);
    }
}

void print_wifi_networks(WifiNetwork networks[], int count) {
    printf("\nDiscovered WiFi Networks:\n");
    printf("SSID                      BSSID            Signal Strength\n");
    printf("---------------------------------------------------------\n");

    for (int i = 0; i < count; i++) {
        printf("%-25s %02x:%02x:%02x:%02x:%02x:%02x      %d dBm\n",
               networks[i].ssid,
               networks[i].bssid[0], networks[i].bssid[1], networks[i].bssid[2],
               networks[i].bssid[3], networks[i].bssid[4], networks[i].bssid[5],
               networks[i].signal_strength);
    }
    printf("---------------------------------------------------------\n");
}