//GEMINI-pro DATASET v1.0 Category: System event logger ; Style: modular
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef __linux__
#include <syslog.h>
#endif

#ifdef _WIN32
#include <windows.h>
#endif

typedef struct event_logger {
    int (*init)(const char *path);
    void (*log)(const char *message);
    void (*close)();
} event_logger_t;

#ifdef __linux__
static int init_syslog(const char *path)
{
    openlog(path, LOG_PID, LOG_USER);
    return 0;
}

static void log_syslog(const char *message)
{
    syslog(LOG_INFO, "%s", message);
}

static void close_syslog()
{
    closelog();
}

static event_logger_t syslog_logger = {
    .init = init_syslog,
    .log = log_syslog,
    .close = close_syslog,
};
#endif

#ifdef _WIN32
static int init_eventlog(const char *path)
{
    HANDLE hEventLog = RegisterEventSource(NULL, path);
    if (hEventLog == NULL)
        return -1;

    return (int)hEventLog;
}

static void log_eventlog(const char *message)
{
    char *buffer = NULL;
    DWORD dwBytesNeeded = 0;

    FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        ERROR_SUCCESS,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
        (LPTSTR)&buffer,
        0,
        NULL);

    dwBytesNeeded = strlen(buffer) + strlen(message) + 2;

    LPSTR pszError = (LPSTR)LocalAlloc(LPTR, dwBytesNeeded);
    if (pszError == NULL)
        return;

    sprintf(pszError, "%s: %s", buffer, message);
    LocalFree(buffer);

    ReportEvent(
        (HANDLE)GetModuleHandle(NULL),
        EVENTLOG_ERROR_TYPE,
        0,
        ERROR_SUCCESS,
        NULL,
        1,
        0,
        &pszError,
        NULL);

    LocalFree(pszError);
}

static void close_eventlog(void)
{
    DeregisterEventSource((HANDLE)GetModuleHandle(NULL));
}

static event_logger_t eventlog_logger = {
    .init = init_eventlog,
    .log = log_eventlog,
    .close = close_eventlog,
};
#endif

static event_logger_t *logger = NULL;

static void init_logger(const char *path)
{
    #ifdef __linux__
    logger = &syslog_logger;
    logger->init(path);
    #endif

    #ifdef _WIN32
    logger = &eventlog_logger;
    logger->init(path);
    #endif
}

static void log_message(const char *message)
{
    if (logger == NULL)
        return;

    logger->log(message);
}

static void close_logger()
{
    if (logger == NULL)
        return;

    logger->close();
}

int main()
{
    time_t t = time(NULL);
    struct tm *tm = localtime(&t);

    char buffer[256];
    strftime(buffer, sizeof(buffer), "%Y-%m-%d %H:%M:%S", tm);

    init_logger("/var/log/myapp.log");
    log_message(buffer);
    close_logger();

    return 0;
}